#!/usr/bin/env python3

import os
import shutil
from errno import EEXIST

from utils.parsing import to_text
from utils.errors import AMMSMLError

__all__ = ['unfrackpath', 'makedirs_safe', 'resolve_path', 'cleanup_tmp_file']


def resolve_path(path, basedir=None):
    """ resolve relative or 'variable' paths """
    if '{{CWD}}' in path:  # allow users to force CWD using 'magic' {{CWD}}
        path = path.replace('{{CWD}}', os.getcwd())

    return unfrackpath(path, follow=False, basedir=basedir)


def unfrackpath(path, follow=True, basedir=None):
    """
    Returns a path that is free of symlinks (if follow=True), environment variables,
    relative path traversals and symbols (~)
    :param path: A byte or text string representing a path to be canonicalized
    :param follow: A boolean to indicate of symlinks should be resolved or not
    :raises UnicodeDecodeError: If the canonicalized version of the path
        contains non-utf8 byte sequences.
    :rtype: A text string
    :returns: An absolute path with symlinks, environment variables, and tilde
              expanded.  Note that this does not check whether a path exists.
    example::
        '$HOME/../../var/mail' becomes '/var/spool/mail'
    """
    b_basedir = to_text(basedir)

    if b_basedir is None:
        b_basedir = to_text(os.getcwd())
    elif os.path.isfile(b_basedir):
        b_basedir = os.path.dirname(b_basedir)

    b_final_path = os.path.expanduser(os.path.expandvars(to_text(path)))

    if not os.path.isabs(b_final_path):
        b_final_path = os.path.join(b_basedir, b_final_path)

    if follow:
        b_final_path = os.path.realpath(b_final_path)

    return to_text(os.path.normpath(b_final_path))


def makedirs_safe(path, mode=None):
    """
    A *potentially insecure* way to ensure the existence of a directory chain. The "safe" in this function's name
    refers only to its ability to ignore `EEXIST` in the case of multiple callers operating on the same part of
    the directory chain. This function is not safe to use under world-writable locations when the first level of the
    path to be created contains a predictable component. Always create a randomly-named element first if there is any
    chance the parent directory might be world-writable (eg, /tmp) to prevent symlink hijacking and potential
    disclosure or modification of sensitive file contents.
    :param path: A byte or text string representing a directory chain to be created
    :param mode: If given, the mode to set the directory to
    :raises AMMSMLError: If the directory cannot be created and does not already exist.
    :raises UnicodeDecodeError: if the path is not decodable in the utf-8 encoding.
    """
    rpath = unfrackpath(path)
    b_rpath = to_text(rpath)
    if not os.path.exists(b_rpath):
        try:
            if mode:
                os.makedirs(b_rpath, mode)
            else:
                os.makedirs(b_rpath)
        except OSError as e:
            if e.errno != EEXIST:
                raise AMMSMLError("Unable to create local directories(%s): %s" % (to_text(rpath), to_text(e)))


def basedir(source):
    """ returns directory from where program is called """
    source = to_text(source)
    dname = None
    if os.path.isdir(source):
        dname = source
    elif source in [None, '', '.']:
        dname = os.getcwd()
    elif os.path.isfile(source):
        dname = os.path.dirname(source)

    if dname:
        # don't follow symlinks for basedir, enables source re-use
        dname = os.path.abspath(dname)

    return to_text(dname)


def cleanup_tmp_file(path, warn=False):
    """
    Removes temporary file or directory. Optionally display a warning if unable
    to remove the file or directory.
    :param path: Path to file or directory to be removed
    :param warn: Whether or not to display a warning when the file or directory
                 cannot be removed
    :return:
    """
    try:
        if os.path.exists(path):
            try:
                if os.path.isdir(path):
                    shutil.rmtree(path)
                elif os.path.isfile(path):
                    os.unlink(path)
            except Exception as e:
                if warn:
                    # Importing here to avoid circular import
                    from ansible.utils.display import Display
                    display = Display()
                    display.display(u'Unable to remove temporary file {0}'.format(to_text(e)))
    except Exception:
        pass
