# GNU Lesser General Public License v3.0+ (see COPYING or https://www.gnu.org/licenses/lgpl-3.0.txt)
# (c) 2019, Christof Schulze <christof.schulze@fau.de>
#
# This file is part of Ammsml
#
# Ammsml is free software: you can redistribute it and/or modify it
# under the terms of the GNU Lesser General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# Ammsml is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU Lesser General Public License for more details.
#
# You should have received a copy of the GNU Lesser General Public License
# along with Ammsml.  If not, see <http://www.gnu.org/licenses/>.

from collections.abc import Sequence


def is_string(seq):
    """Identify whether the input has a string-like type."""
    return isinstance(seq, str)


def is_iterable(seq, include_strings=False):
    """Identify whether the input is an iterable."""
    if not include_strings and is_string(seq):
        return False

    try:
        iter(seq)
        return True
    except TypeError:
        return False


def is_sequence(seq, include_strings=False):
    """
    Identify whether the input is a sequence.
    Strings are not sequences here,
    unless ``include_string`` is ``True``.
    Non-indexable things are never of a sequence type.
    """
    if not include_strings and is_string(seq):
        return False

    return isinstance(seq, Sequence)


def count(seq):
    """
    Returns a dictionary with the number of appearances of each element of the iterable.

    Resembles the `collections.Counter` class functionality, which is deprecated
    since version 3.3, will be removed in version 3.9
    It is meant to be replaced when the we find something equivalent in maybe collections.abc

    this code also runs on Python 2.6.* where collections.Counter is not available.

    """
    if not is_iterable(seq):
        raise Exception('Argument provided  is not an iterable')

    if not is_iterable(seq):
        raise Exception('Argument provided  is not an iterable')
    counters = dict()
    for elem in seq:
        counters[elem] = counters.get(elem, 0) + 1
    return counters
