# GNU Lesser General Public License v3.0+ (see COPYING or https://www.gnu.org/licenses/lgpl-3.0.txt)
# (c) 2019, Christof Schulze <christof.schulze@fau.de>
#
# This file is part of Ammsml
#
# Ammsml is free software: you can redistribute it and/or modify it
# under the terms of the GNU Lesser General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# Ammsml is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU Lesser General Public License for more details.
#
# You should have received a copy of the GNU Lesser General Public License
# along with Ammsml.  If not, see <http://www.gnu.org/licenses/>.

import sys
import re

from ammsml.config import constants as C

IS_COLOR: bool = True

if not hasattr(sys.stdout, 'isatty') or not sys.stdout.isatty():
    IS_COLOR = False
else:
    try:
        import curses
        curses.setupterm()
        if curses.tigetnum('colors') < 0:
            IS_COLOR = False
    except ImportError:
        # curses library was not found
        pass
    except curses.error:
        # curses returns an error (e.g. could not find terminal)
        IS_COLOR = False

# TODO implement some Config entry for Color
# if C.IS_FORCE_COLOR:
#     IS_COLOR = True

# --- begin "pretty"
#
# pretty - A miniature library that provides a Python print and stdout
# wrapper that makes colored terminal text easier to use (e.g. without
# having to mess around with ANSI escape sequences). This code is public
# domain - there is no license except that you must leave this header.
#
# Copyright (C) 2008 Brian Nez <thedude at bri1 dot com>
#
# http://nezzen.net/2008/06/23/colored-text-in-python-using-ansi-escape-sequences/


codeCodes = {
    'black': '0;30', 'bright gray': '0;37',
    'blue': '0;34', 'white': '1;37',
    'green': '0;32', 'bright blue': '1;34',
    'cyan': '0;36', 'bright green': '1;32',
    'red': '0;31', 'bright cyan': '1;36',
    'purple': '0;35', 'bright red': '1;31',
    'yellow': '0;33', 'bright purple': '1;35',
    'dark gray': '1;30', 'bright yellow': '1;33',
    'magenta': '0;35', 'bright magenta': '1;35',
    'normal': '0',
}


def parsecolor(color):
    """SGR parameter string for the specified color name."""
    matches = re.match(r"color(?P<color>[0-9]+)"
                       r"|(?P<rgb>rgb(?P<red>[0-5])(?P<green>[0-5])(?P<blue>[0-5]))"
                       r"|gray(?P<gray>[0-9]+)", color)
    if not matches:
        return codeCodes[color]
    if matches.group('color'):
        return '38;5;%d' % int(matches.group('color'))
    if matches.group('rgb'):
        return '38;5;%d' % (16 + 36 * int(matches.group('red')) +
                            6 * int(matches.group('green')) +
                            int(matches.group('blue')))
    if matches.group('gray'):
        return '38;5;%d' % (232 + int(matches.group('gray')))


def stringc(text: str, color) -> str:
    """String in color."""

    if IS_COLOR:
        color_code = parsecolor(color)
        return "\n".join(["\033[%sm%s\033[0m" % (color_code, t) for t in text.split('\n')])
    else:
        return text


def colorize(lead, num, color):
    """ Print 'lead' = 'num' in 'color' """
    s = "%s=%-4s" % (lead, str(num))
    if num != 0 and IS_COLOR and color is not None:
        s = stringc(s, color)
    return s


def hostcolor(host: str, stats: list, color: bool = True) -> str:
    if IS_COLOR and color:
        if stats['failures'] != 0 or stats['unreachable'] != 0:
            return "%-37s" % stringc(host, C.COLOR_ERROR)
        elif stats['changed'] != 0:
            return "%-37s" % stringc(host, C.COLOR_CHANGED)
        else:
            return "%-37s" % stringc(host, C.COLOR_OKGREEN)
    return "%-26s" % host
