# GNU Lesser General Public License v3.0+ (see COPYING or https://www.gnu.org/licenses/lgpl-3.0.txt)
# (c) 2019, Christof Schulze <christof.schulze@fau.de>
#
# This file is part of Ammsml
#
# Ammsml is free software: you can redistribute it and/or modify it
# under the terms of the GNU Lesser General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# Ammsml is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU Lesser General Public License for more details.
#
# You should have received a copy of the GNU Lesser General Public License
# along with Ammsml.  If not, see <http://www.gnu.org/licenses/>.

BOOLEANS_TRUE = frozenset(('y', 'yes', 'on', '1', 'true', 't', 1, 1.0, True))
BOOLEANS_FALSE = frozenset(('n', 'no', 'off', '0', 'false', 'f', 0, 0.0, False))
BOOLEANS = BOOLEANS_TRUE.union(BOOLEANS_FALSE)


def to_text(obj, encoding='utf-8'):
    """
    """

    if isinstance(obj, str):
        return obj

    if isinstance(obj, bytes):
        # Note: We don't need special handling for surrogate_then_replace
        # because all bytes will either be made into surrogates or are valid
        # to decode.
        return obj.decode(encoding)

    try:
        value = str(obj)

    except UnicodeError:
        try:
            value = repr(obj)
        except UnicodeError:
            # Giving up
            raise TypeError('Invalid value %s for to_text\'s nonstring parameter' % value)

    return to_text(value)


def boolean(value):
    if isinstance(value, bool):
        return value

    normalized_value = value
    if isinstance(value, (str, bytes)):
        normalized_value = to_text(value).lower().strip()

    if normalized_value in BOOLEANS_TRUE:
        return True
    elif normalized_value in BOOLEANS_FALSE:
        return False

    raise TypeError("The value '%s' is not a valid boolean.  Valid booleans include: %s" % (
                    to_text(value), ', '.join(repr(i) for i in BOOLEANS)))


def unquote(data):
    """
    removes first and last quotes from a string,
    if the string starts and ends with the same quotes
    """
    if is_quoted(data):
        return data[1:-1]

    return data


def is_quoted(data):
    return len(data) > 1 and data[0] == data[-1] and data[0] in ('"', "'") and data[-2] != '\\'





