
import os
import time
import yaml

from ammsml.utils.path import unfrackpath, resolve_path

__all__ = ['gitinfo']


def _git_repo_info(repo_path):
    """
    returns a string containing git branch, commit id and commit date
    """
    result = None
    if os.path.exists(repo_path):
        # Check if the .git is a file. If it is a file, it means that we are in a submodule structure.
        if os.path.isfile(repo_path):
            try:
                gitdir = yaml.safe_load(open(repo_path)).get('gitdir')
                # There is a possibility the .git file to have an absolute path.
                if os.path.isabs(gitdir):
                    repo_path = gitdir
                else:
                    repo_path = os.path.join(repo_path[:-4], gitdir)
            except (IOError, AttributeError):
                return ''
        with open(os.path.join(repo_path, "HEAD")) as f:
            line = f.readline().rstrip("\n")
            if line.startswith("ref:"):
                branch_path = os.path.join(repo_path, line[5:])
            else:
                branch_path = None
        if branch_path and os.path.exists(branch_path):
            branch = '/'.join(line.split('/')[2:])
            with open(branch_path) as f:
                commit = f.readline()[:10]
        else:
            # detached HEAD
            commit = line[:10]
            branch = 'detached HEAD'
            branch_path = os.path.join(repo_path, "HEAD")

        date = time.localtime(os.stat(branch_path).st_mtime)
        if time.daylight == 0:
            offset = time.timezone
        else:
            offset = time.altzone
        result = "({0} {1}) last updated {2} (GMT {3:+04d})".format(branch, commit,
                                                                    time.strftime("%Y/%m/%d %H:%M:%S", date),
                                                                    int(offset / -36))
    else:
        result = ''
    return result


def gitinfo(path=None):
    if path is None:
        # for use in ammsml
        basedir = os.path.join(os.path.dirname(__file__))
    else:
        basedir = path
    # print('basedir', basedir)
    repo_path = unfrackpath(os.path.join(basedir, '.git'))
    # print("repo_path", repo_path)
    result = _git_repo_info(repo_path)

    submodules = os.path.join(basedir, '.gitmodules')
    if not os.path.exists(submodules):
        return result

    with open(submodules) as f:
        for line in f:
            tokens = line.strip().split(' ')
            if tokens[0] == 'path':
                submodule_path = tokens[2]
                submodule_info = _git_repo_info(os.path.join(basedir, submodule_path, '.git'))
                if not submodule_info:
                    submodule_info = ' not found - use git submodule update --init ' + submodule_path
                result += "\n  {0}: {1}".format(submodule_path, submodule_info)
    return result


if __name__ == '__main__':
    """
    Checks branch and version (commit) and last modified date from a git repository.
    Shows (detached HEAD ref: refs/)  on modified repository,
    and the date of last commit.
    FIXME does not work in terminal $>python git.py 
    """
    print('Testing Git\n===========')
    print('None', __file__)
    print('path', __file__)
    print(gitinfo(), '\n---------')

    testdir1 = resolve_path(os.path.join(os.path.dirname(__file__), '..', 'config'))
    print('path', os.path.join(os.path.dirname(__file__), '..', 'config'))
    print('path', testdir1)
    print(gitinfo(testdir1), '\n---------')

    testdir2 = resolve_path(os.path.join(os.path.dirname(__file__), '..', '..', '..', 'python', 'file_utils'))
    print(os.path.dirname(__file__))
    print('path', os.path.join(os.path.dirname(__file__), '..', '..', '..', 'python', 'file_utils'))
    print('path', testdir2)
    print(gitinfo(testdir2), '\n---------')
