#!/usr/bin/env python3


def to_text(obj, encoding='utf-8'):
    """
    """

    if isinstance(obj, str):
        return obj

    if isinstance(obj, bytes):
        # Note: We don't need special handling for surrogate_then_replace
        # because all bytes will either be made into surrogates or are valid
        # to decode.
        return obj.decode(encoding)

    try:
        value = str(obj)

    except UnicodeError:
        try:
            value = repr(obj)
        except UnicodeError:
            # Giving up
            raise TypeError('Invalid value %s for to_text\'s nonstring parameter' % value)

    return to_text(value)


BOOLEANS_TRUE = frozenset(('y', 'yes', 'on', '1', 'true', 't', 1, 1.0, True))
BOOLEANS_FALSE = frozenset(('n', 'no', 'off', '0', 'false', 'f', 0, 0.0, False))
BOOLEANS = BOOLEANS_TRUE.union(BOOLEANS_FALSE)


def boolean(value):
    if isinstance(value, bool):
        return value

    normalized_value = value
    if isinstance(value, (str, bytes)):
        normalized_value = to_text(value).lower().strip()

    if normalized_value in BOOLEANS_TRUE:
        return True
    elif normalized_value in BOOLEANS_FALSE:
        return False

    raise TypeError("The value '%s' is not a valid boolean.  Valid booleans include: %s" % (to_text(value), ', '.join(repr(i) for i in BOOLEANS)))


def is_quoted(data):
    return len(data) > 1 and data[0] == data[-1] and data[0] in ('"', "'") and data[-2] != '\\'


def unquote(data):
    """
    removes first and last quotes from a string,
    if the string starts and ends with the same quotes
    """
    if is_quoted(data):
        return data[1:-1]

    return data
