# GNU Lesser General Public License v3.0+ (see COPYING or https://www.gnu.org/licenses/lgpl-3.0.txt)
# (c) 2019, Christof Schulze <christof.schulze@fau.de>
#
# This file is part of Ammsml
#
# Ammsml is free software: you can redistribute it and/or modify it
# under the terms of the GNU Lesser General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# Ammsml is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU Lesser General Public License for more details.
#
# You should have received a copy of the GNU Lesser General Public License
# along with Ammsml.  If not, see <http://www.gnu.org/licenses/>.

from collections.abc import Mapping, MutableSequence, Set


__all__ = ['AMMSMLUnsafe', 'wrap_var']


class AMMSMLUnsafe(object):
    __UNSAFE__ = True


class AMMSMLUnsafeBytes(bytes, AMMSMLUnsafe):
    pass


class AMMSMLUnsafeText(str, AMMSMLUnsafe):
    pass


def _wrap_dict(v):
    for k in v.keys():
        if v[k] is not None:
            v[wrap_var(k)] = wrap_var(v[k])
    return v


def _wrap_list(v):
    for idx, item in enumerate(v):
        if item is not None:
            v[idx] = wrap_var(item)
    return v


def _wrap_set(v):
    return set(item if item is None else wrap_var(item) for item in v)


def wrap_var(v):
    if isinstance(v, AMMSMLUnsafe):
        return v
    if isinstance(v, Mapping):
        v = _wrap_dict(v)
    elif isinstance(v, MutableSequence):
        v = _wrap_list(v)
    elif isinstance(v, Set):
        v = _wrap_set(v)
    elif isinstance(v, bytes):
        v = AMMSMLUnsafeBytes(v)
    elif isinstance(v, str):
        v = AMMSMLUnsafeText(v)

    return v
