# GNU Lesser General Public License v3.0+ (see COPYING or https://www.gnu.org/licenses/lgpl-3.0.txt)
# (c) 2019, Christof Schulze <christof.schulze@fau.de>
#
# This file is part of Ammsml
#
# Ammsml is free software: you can redistribute it and/or modify it
# under the terms of the GNU Lesser General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# Ammsml is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU Lesser General Public License for more details.
#
# You should have received a copy of the GNU Lesser General Public License
# along with Ammsml.  If not, see <http://www.gnu.org/licenses/>.

BOOLEANS_TRUE = frozenset(('y', 'yes', 'on', '1', 'true', 't', 1, 1.0, True))
BOOLEANS_FALSE = frozenset(('n', 'no', 'off', '0', 'false', 'f', 0, 0.0, False))
BOOLEANS = BOOLEANS_TRUE.union(BOOLEANS_FALSE)

# The error handler to use if the byte string is not decodable using the specified encoding.
decoding_error = 'surrogateescape'


def to_text(obj, encoding='utf-8', nonstring: str = 'simplerepr'):
    """
    Make sure that a string is a text string

    :param obj:  An object to make sure is a text string.  In most cases this
        will be either a text string or a byte string.  However, with
        ``nonstring='simplerepr'``, this can be used as a traceback-free
        version of ``str(obj)``.
    :param encoding:  The encoding to use to transform from a byte string to
        a text string.  Defaults to using 'utf-8'.

    :param nonstring: The strategy to use if a nonstring is specified in
        ``obj``.  Default is 'simplerepr'.  Valid values are:

        :simplerepr: The default.  This takes the ``str`` of the object and
            then returns the text version of that string.
        :empty: Return an empty text string
        :passthru: Return the object passed in
        :strict: Raise a :exc:`TypeError`

    :return: Typically this returns a text string.  If a nonstring object is
        passed in this may be a different type depending on the strategy
        specified by nonstring.  This will never return a byte string.
    """

    if isinstance(obj, str):
        return obj

    if isinstance(obj, bytes):
        # Note: We don't need special handling for surrogateescape because
        # all bytes will either be made into surrogates or are valid to decode.
        return obj.decode(encoding, decoding_error)

    # Note: We do these last even though we have to call to_text again on the
    # value because we're optimizing the common case
    if nonstring == 'simplerepr':
        try:
            value = str(obj)

        except UnicodeError:
            try:
                value = repr(obj)
            except UnicodeError:
                # Giving up
                return ''
    elif nonstring == 'passthru':
        return obj
    elif nonstring == 'empty':
        return ''
    elif nonstring == 'strict':
        raise TypeError('obj must be a string type')
    else:
        raise TypeError('Invalid value %s for to_text\'s nonstring parameter' % nonstring)

    return to_text(value, encoding, decoding_error)


def boolean(value):
    if isinstance(value, bool):
        return value

    normalized_value = value
    if isinstance(value, (str, bytes)):
        normalized_value = to_text(value).lower().strip()

    if normalized_value in BOOLEANS_TRUE:
        return True
    elif normalized_value in BOOLEANS_FALSE:
        return False

    raise TypeError("The value '%s' is not a valid boolean.  Valid booleans include: %s" % (
                    to_text(value), ', '.join(repr(i) for i in BOOLEANS)))


def to_bytes(obj, encoding='utf-8', nonstring='simplerepr'):
    """
    Make sure that a string is a byte string
    :param obj: An object to make sure is a byte string.  In most cases this
        will be either a text string or a byte string.  However, with
        ``nonstring='simplerepr'``, this can be used as a traceback-free
        version of ``str(obj)``.
    :param encoding: The encoding to use to transform from a text string to
        a byte string.  Defaults to using 'utf-8'.
    :param nonstring: The strategy to use if a nonstring is specified in
        ``obj``.  Default is 'simplerepr'.  Valid values are:
        :simplerepr: The default.  This takes the ``str`` of the object and
            then returns the bytes version of that string.
        :empty: Return an empty byte string
        :passthru: Return the object passed in
        :strict: Raise a :exc:`TypeError`
    :return: Typically this returns a byte string. If a nonstring object is
        passed in this may be a different type depending on the strategy
        specified by nonstring.  This will never return a text string.
    .. note:: If passed a byte string, this function does not check that the
        string is valid in the specified encoding. If it's important that the
        byte string is in the specified encoding do::
            encoded_string = to_bytes(to_text(input_string, 'latin-1'), 'utf-8')
    """

    if isinstance(obj, bytes):
        return obj

    # We're given a text string
    # If it has surrogates, we know because it will decode
    if isinstance(obj, str):
        try:
            # Try this first as it's the fastest
            return obj.encode(encoding)
        except UnicodeEncodeError:
            raise

    # Note: We do these last even though we have to call to_bytes again on the
    # value because we're optimizing the common case
    if nonstring == 'simplerepr':
        try:
            value = str(obj)
        except UnicodeError:
            try:
                value = repr(obj)
            except UnicodeError:
                # Giving up
                return to_bytes('')
    elif nonstring == 'passthru':
        return obj
    elif nonstring == 'empty':
        return to_bytes('')
    elif nonstring == 'strict':
        raise TypeError('obj must be a string type')
    else:
        raise TypeError('Invalid value %s for to_bytes\' nonstring parameter' % nonstring)

    return to_bytes(value, encoding)


def unquote(data):
    """
    removes first and last quotes from a string,
    if the string starts and ends with the same quotes
    """
    if is_quoted(data):
        return data[1:-1]

    return data


def is_quoted(data):
    return len(data) > 1 and data[0] == data[-1] and data[0] in ('"', "'") and data[-2] != '\\'


def humanize_time(d) -> str:
    """
    Return a human-friendly description of elapsed time
    :param d:
    :return:
    """
    human = ""
    if d >= 3600:
        human += "%dh" % (int(d) // 3600)
        d %= 3600
    if d >= 60:
        human += "%dm" % (int(d) // 60)
        d %= 60
    human += ("%.3fs" % d)
    return human
